#include "stdafx.h"
#include "Debug.h"
#include "Common.h"
#include "Application.h"

#include "TLR_Results.h"
#include "EtherCATMaster_AP_Public.h"
#include "EtherCATMaster_Public.h"
typedef char STRING;
#include "EtherCATMaster_DiagStructDef.h"
#include "rcX_Public.h"

/*************************************************************************************************
 * @brief This method handles keys pressed by the user.
 * 
 * @param ptApp Pointer to application data.
 * @param iKey is the key pressed.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandleKey(APPLICATION_T*  ptApp, TLR_INT iKey)
{
  switch(iKey)
  {
    case 'h':
		App_ShowHelp();
		break;

	case 'u':
		App_HandleSdoUploadReq(ptApp);
		break;

	case 'g':
		App_GetHandles(ptApp);
		break;
	default:
		DEBUG("Unknown key  (%c)  pressed.\n", iKey);
		break;
  }
}


void App_HandleSdoUploadReq(APPLICATION_T* ptApp)
{
	ETHERCAT_MASTER_PACKET_SDO_UPLOAD_REQ_T tPck;

	char cBuf[128];

	memset(&tPck, 0, sizeof(tPck));

	tPck.tHead.ulCmd  = ETHERCAT_MASTER_CMD_SDO_UPLOAD_REQ;
	tPck.tHead.ulDest = 0x20;
	tPck.tHead.ulLen  = sizeof(tPck.tData);

	DEBUG("Specify Node Address, e. g. \"256\"  ");
	fgets(cBuf, 128, stdin);
	tPck.tData.ulNodeId = atol(cBuf);

	DEBUG("Specify Index, e.g. \"4096\"  ");
	fgets(cBuf, 128, stdin);
	tPck.tData.ulIndex = atol(cBuf);

	DEBUG("Specify Subindex, e.g. \"1\"  ");
	fgets(cBuf, 128, stdin);
	tPck.tData.ulSubIndex = atol(cBuf);

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
    }
}


void App_GetHandles(APPLICATION_T* ptApp)
{
	RCX_PACKET_GET_SLAVE_HANDLE_REQ_T tPck;

	memset(&tPck, 0, sizeof(tPck));

	tPck.tHead.ulCmd  = RCX_GET_SLAVE_HANDLE_REQ;
	tPck.tHead.ulDest = 0x20;
	tPck.tHead.ulLen  = sizeof(tPck.tData);

	tPck.tData.ulParam = RCX_LIST_CONF_SLAVES;


	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
	}
}

void App_GetSlaveState(APPLICATION_T* ptApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt)
{
	/* read MasterStatusBlock and printout configured slaves and active slaves */
	NETX_COMMON_STATUS_BLOCK tCommonStatus = {0};
	xChannelCommonStatusBlock(ptApp->tCommon.hChannel, CIFX_CMD_READ_DATA, 0, sizeof(tCommonStatus), &tCommonStatus);

	if (ptApp->tCommon.ulConfigSlaveAmount != tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfConfigSlaves)
	{
		DEBUG("Configured Slaves amount changed from %u to %u.\n", ptApp->tCommon.ulConfigSlaveAmount, tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfConfigSlaves);
		ptApp->tCommon.ulConfigSlaveAmount = tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfConfigSlaves;
	}

	if (ptApp->tCommon.ulActiveSlaveAmount != tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfActiveSlaves)
	{
		DEBUG("Active Slave amount changed from %u to %u.\n", ptApp->tCommon.ulActiveSlaveAmount, tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfActiveSlaves);
		ptApp->tCommon.ulActiveSlaveAmount = tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfActiveSlaves;
	}

	if (ptApp->tCommon.ulFaultySlaveAmount != tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfDiagSlaves)
	{
		DEBUG("Faulty Slaves amount changed from %u to %u.\n", ptApp->tCommon.ulFaultySlaveAmount, tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfDiagSlaves);
		ptApp->tCommon.ulFaultySlaveAmount = tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfDiagSlaves;
	}
}

void App_GetDeviceInfo(APPLICATION_T* ptApp, TLR_UINT32 ulHandle)
{
	RCX_PACKET_GET_SLAVE_CONN_INFO_REQ_T tPck;
	memset(&tPck, 0, sizeof(tPck));

	tPck.tHead.ulCmd  = RCX_GET_SLAVE_CONN_INFO_REQ;
	tPck.tHead.ulDest = 0x20;
	tPck.tHead.ulLen  = sizeof(tPck.tData);

	tPck.tData.ulHandle = ulHandle;

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
	}
}



/*************************************************************************************************
 * @brief This method handles all packets which will be received by CoE application.
 * 
 * @param ptApp Pointer to application data.
 * @param ptPacket Pointer to CifX packet.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandlePacket(APPLICATION_T*  ptApp, CIFX_PACKET* ptPacket)
{
  TLR_PACKET_HEADER_T* ptPck = (TLR_PACKET_HEADER_T*)ptPacket;

  switch(ptPck->ulCmd)
  {
  case RCX_GET_SLAVE_HANDLE_CNF:
	  App_HandleGetSlaveHandleCnf(ptApp, ptPacket);
	  break;

  case RCX_GET_SLAVE_CONN_INFO_CNF:
	  App_HandleGetSlaveConnInfoCnf(ptApp, ptPacket);
	  break;

  case ETHERCAT_MASTER_CMD_SDO_UPLOAD_CNF:
	  App_HandleSdoUploadCnf(ptApp, ptPacket);
	  break;

  default:
      DEBUG("Unknown packet. ulCmd=0x%x\n", ptPacket->tHeader.ulCmd);
      break;
  }
}

/******************************************************************/
/*          functions handling packets send by firmware           */
/******************************************************************/


void App_HandleGetSlaveHandleCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	RCX_PACKET_GET_SLAVE_HANDLE_CNF_T* ptPck = (RCX_PACKET_GET_SLAVE_HANDLE_CNF_T*)ptPacket;

	/* calculate the amount of handles inside this packet */
	/* the first handle is ptPck->tData.aulHandle[0] */
	/* the second is (and so on) ptPck->tData.aulHandle[1] */
	TLR_UINT32 ulHandleCnt = ptPck->tHead.ulLen - sizeof(ptPck->tData);
	ulHandleCnt = ulHandleCnt / 4;
	ulHandleCnt++;
	TLR_UINT32 ulTmp;

	if (16 > ulHandleCnt)
	{
		for(ulTmp = 0; ulTmp < ulHandleCnt; ulTmp++)
		{
			/* get device infor for the specific handle */
			App_GetDeviceInfo(ptApp, ptPck->tData.aulHandle[ulTmp]);
		}
	}
	else
	{
		DEBUG("%u devices configured. This needs more precise handling as only 16 message can be put in the mailbox per time.\n", ulHandleCnt);
	}

}

void App_HandleGetSlaveConnInfoCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	RCX_PACKET_GET_SLAVE_CONN_INFO_CNF_T* ptPck = (RCX_PACKET_GET_SLAVE_CONN_INFO_CNF_T*)ptPacket;
	ETHERCAT_MASTER_DIAG_GET_SLAVE_DIAG_T* ptSlaveDiag;

	if (ptPck->tData.ulStructID == 5938)
	{
		ptSlaveDiag = (ETHERCAT_MASTER_DIAG_GET_SLAVE_DIAG_T*)(&ptPck->tData +1);
		DEBUG("Handle %u: StationAddress: %d, \n\t AutoIncAddress: 0x%X,  Name %s\n", 
			ptPck->tData.ulHandle,
			ptSlaveDiag->ulStationAddress,
			ptSlaveDiag->ulAutoIncAddress,
			ptSlaveDiag->szSlaveName);
	}
	else
	{
		DEBUG("unexpected StructId %u returned.\n", ptPck->tData.ulStructID);
	}
}


void App_HandleSdoUploadCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	ETHERCAT_MASTER_PACKET_SDO_UPLOAD_CNF_T* ptPck = (ETHERCAT_MASTER_PACKET_SDO_UPLOAD_CNF_T*)ptPacket;
	if (TLR_E_ETHERCAT_MASTER_COE_NO_MBX_SUPPORT == ptPck->tHead.ulSta)
	{
		DEBUG("SDO Upload: Slave does not supports Mailbox!\n");
	}
	else if (TLR_S_OK != ptPck->tHead.ulSta)
	{
		DEBUG("SDO Upload: error returned 0x%X returned.\n", ptPck->tHead.ulSta);
	}
	else
	{
		/* copy 4 byte payload into temp variable and print it */
		UINT32 ulTemp;
		memcpy(&ulTemp, ptPck->tData.abSdoData, sizeof(ulTemp));
		DEBUG("SDO Upload: nodeID: %d, index: 0x%X, subindex: 0x%X, value: %d\n", ptPck->tData.ulNodeId, ptPck->tData.ulIndex, ptPck->tData.ulSubIndex, ulTemp);
	}
}


/*************************************************************************************************
 * @brief This function creates all resources for the application.
 * It allocates memory for the application data and returns a pointer to it.
 * App_FreeResources() must be called in order to free resources again.
 * 
 * @param pptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_CreateResources(APPLICATION_T** pptApp)
{
  DEBUG("Allocating resources...\n");

  /* return value of function */
  TLR_RESULT tResult      = TLR_S_OK;

  /* at the end of the function we will return the pointer */
  *pptApp = NULL;

  /* allocate memory for application data */
  APPLICATION_T* ptApp = (APPLICATION_T*)malloc(sizeof(APPLICATION_T));
  memset(ptApp, 0, sizeof(APPLICATION_T));
  
  /* allocate memory for read / write buffers */
  ptApp->tCommon.ulReadBufferSize  = 8 * sizeof(TLR_UINT8);
  ptApp->tCommon.ulWriteBufferSize = 8 * sizeof(TLR_UINT8);

  ptApp->tCommon.pabReadBuffer  =  (TLR_UINT8*) malloc(ptApp->tCommon.ulReadBufferSize);
  ptApp->tCommon.pabWriteBuffer =  (TLR_UINT8*) malloc(ptApp->tCommon.ulWriteBufferSize);

  /* initialize the read and write buffer with zero */
  memset(ptApp->tCommon.pabReadBuffer,  0, ptApp->tCommon.ulReadBufferSize);
  memset(ptApp->tCommon.pabWriteBuffer, 0, ptApp->tCommon.ulWriteBufferSize);

  /* return the pointer */
  *pptApp = ptApp;

  DEBUG("Successful.\n");
  return tResult;
}

/*************************************************************************************************
 * @brief This function frees all resources created by App_CreateResources(). 
 * 
 * @param pptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_FreeResources(APPLICATION_T* ptApp)
{
  DEBUG("Free resources...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* free buffer resources */
  free(ptApp->tCommon.pabReadBuffer);
  free(ptApp->tCommon.pabWriteBuffer);

  /* free application data container */
  free(ptApp);

  DEBUG("Successful.\n");
  return tResult;
}

/*************************************************************************************************
 * @brief This method handles the cyclic process data exchange.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandleProcessData(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* cyclic io handing happens here */
  /* read RTD-Data */
  tResult = xChannelIORead(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulReadBufferSize, ptApp->tCommon.pabReadBuffer, ptApp->tCommon.ulReadBufferSize);

  /* the action depends on the return value we have received */
  if ((CIFX_DEV_EXCHANGE_FAILED == tResult) || (CIFX_DEV_EXCHANGE_TIMEOUT == tResult))
  {
    DEBUG("Read failed with 0x%08x\n", tResult);
  }
  else if (CIFX_DEV_NO_COM_FLAG == tResult)
  {
    //DEBUG("Read failed. Device is not communicating\n");
  }
  else if (CIFX_NO_ERROR == tResult)
  {
    //DEBUG("Read succeeded. Copy Inputs back to Outputs.\n");
    
    /* we copy the memory we have read to the memory we want to send, */
    /* because we just want to mirror the data */
    memcpy (ptApp->tCommon.pabWriteBuffer, ptApp->tCommon.pabReadBuffer, ptApp->tCommon.ulWriteBufferSize);     

    // write IO-Data 
    tResult = xChannelIOWrite(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulWriteBufferSize, ptApp->tCommon.pabWriteBuffer, ptApp->tCommon.ulWriteBufferSize);
  }
  else
  {
    /* received unexpected failure */
    DEBUG("Read failed unexpected with 0x%08x\n", tResult);
  }

}

/*************************************************************************************************
 * @brief This function initializes the application. 
 * Objects will be creates and services will be registered inside.
 * App_Finalize() must be called in order to achieve a friendly shutdown of application.
 * 
 * @param ptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Initialize(APPLICATION_T* ptApp)
{
  DEBUG("Initializing application...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* register on firmware for indications */
  tResult = App_SendRecvEmptyPkt(ptApp, RCX_REGISTER_APP_REQ);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }  
  
  return tResult;
}


/*************************************************************************************************
 * @brief This method finalizes the application. 
 * It returns handles, aso.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Finalize(APPLICATION_T* ptApp)
{
  DEBUG("Shutdown application...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* at first we empty the queue, than we start with the shutdown sequence below */
  while(TLR_S_OK == (tResult = xChannelGetPacket(ptApp->tCommon.hChannel, sizeof(tRecvPkt), &tRecvPkt, ptApp->tCommon.ulTimeout)))
  {
    App_HandlePacket(ptApp, &tRecvPkt);
  }

  /* unregister on firmware */
  App_SendRecvEmptyPkt(ptApp, RCX_UNREGISTER_APP_REQ);

  /* perform ChannelInit on firmware to stop any communication and delete configuration */
  App_SendRecvEmptyPkt(ptApp, RCX_CHANNEL_INIT_REQ);

  return tResult;
}

/*************************************************************************************************
 * @brief This function prompts a debug message at application startup.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_PromptIntro(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  DEBUG("\n");
  DEBUG("**********************************************************\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   Configuration example for EtherCAT Master            *\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   Copyright (c) Hilscher GmbH. All Rights Reserved.    *\n");
  DEBUG("*                                                        *\n");
  DEBUG("**********************************************************\n");
  DEBUG("\n");

  return tResult;
}

